import Sse from "./modules/sse/index";
import Rtc from "./modules/webrtc/rtc";
import recorder from "./modules/recorder";

/**  error: 关闭直播 */
const ERROR_CODES = code => [9, 99, 50, 5, 99999999].includes(code);

class LiveExample {
  constructor() {
    // 声网rtc实例
    this.rtcClient = null;

    // 麦克风接管socket实例
    this.sseAudioClient = null;

    // 麦克风接管实例
    this.audioOnlineInstance = null;

    // 直播参数 - channelId
    this.channelId = undefined;

    // 直播参数 - sessionId
    this.sessionId = null;

    // 麦克风接管参数 - sessionToken
    this.sessionToken = null;
    
    // 直播状态， false - 未直播， true - 直播中
    this.isConnect = false;
  }

  /**
   * 新建直播间
   *
   * @param {String} param - 直播视频生成参数（该参数为json转义后的字符串）
   *
   * @return {String} res.data.sessionId - 会话的唯一标识
   * @return {String} res.data.sessionToken - 麦克风接管使用的token信息
   */
  handleCreate(param) {
    // 创建声网rtc实例
    this.rtcClient = new Rtc();

    this.$axios
      .post("/api/2dvh/v2/material/live/channel/create", {
        param: param,
      })
      .then((res) => {
        if (res.code === 0) {
          const { sessionToken, sessionId } = res.data;
          this.sessionId = sessionId;
          this.sessionToken = sessionToken;
          this.isConnect = true;
          this.sendHeartbeat();
        }

        this.handleLiveRes(res);
      });
  }

  /**
   * 处理直播状态返回数据
   *
   * @param {Integer} res.code - 服务状态码
   * @param {Integer} res.data.status - 任务状态：32:视频流创建中, 35:视频流创建完成, 其他：参考文档
   * @param {String} res.data.appId - APP ID （状态为32无此值）
   * @param {String} res.data.channelId - Channel ID  （状态为32无此值）
   *
   */
  handleLiveRes(res) {
    const { code, message, data } = res;

    // 错误
    if (code !== 0 || ERROR_CODES(data?.status)) {
      this.sessionId = null;
      return;
    }

    const { status } = data;

    // 状态：直播已关闭
    if (status === 5 || status === 50) {
      this.isConnect = false;
      return;
    }

    // 状态：直播间创建中
    if (status === 32) {
      setTimeout(() => {
        this.getLiveStatus();
      }, 3000);
      return;
    }

    // 状态：直播间创建完成
    if (status === 35) {
      const { rtcAppId, rtcChannelId, rtcToken, rtcUid } = data;
      this.channelId = rtcChannelId;
      this.rtcClient.init({
        appId: rtcAppId,
        channel: rtcChannelId,
        token: rtcToken,
        customerUid: rtcUid,
        remotePlayerContainer: document.getElementById("liveVideo"),
        sessionId: this.sessionId
      });

      // 监听是否成功进入频道
      this.rtcClient.addEventListener(
        "user-published",
        () => {

          // 推流后的处理
        },
        false
      );
    }
  }

  /**
   * 查询会话状态
   *
   * @param {String} sessionId - 会话的唯一标识
   *
   * @return {Integer} res.data.status - 任务状态：32:视频流创建中, 35:视频流创建完成, 其他：参考文档
   * 
   */
  getLiveStatus() {
    this.$axios
      .get(
        `/api/2dvh/v1/material/live/channel/stat?sessionId=${this.sessionId}`
      )
      .then(this.handleLiveRes)
      .catch(e => {
        // 网络错误，尝试补发
        if (e.message === "Network Error") {
          setTimeout(() => {
            this.getLiveStatus();
          }, 3000);
        } else {
          this.isConnect = false;
          this.sessionId = null;
        }
      });
  }

  /**
   * 开启直播
   *
   * @param {String} sessionId - 会话的唯一标识
   */
  handleStart(sessionId) {
    this.$axios
      .post("/api/2dvh/v1/material/live/channel/start", {
        sessionId: sessionId,
      })
      .then((res) => res.data)
      .then((data) => {
        const { code, message } = res;
        if (code !== 0) {
          this.$message.error(message);
          return;
        }

        // 直播已成功开启，初始化麦克风接管socket
        this.initAudioSSE();
      });
  }

  /**
   * 直播状态监测(直播期间保持心跳定期发送, 建议60s)
   *
   */
  sendHeartbeat() {
    setTimeout(() => {
      if (this.isConnect) {
        this.$axios
          .post(`/api/2dvh/v1/material/live/channel/heartbeat`, {
            sessionId: this.sessionId,
          })
          .then(res => {
            const { code } = res;
            if (code !== 0) {
              throw new Error(code);
            }
          })
          .catch(e => {
            console.warn("ping message fail");
          })
          .finally(() => {
            this.sendHeartbeat();
          });
      }
    }, 60 * 1000);
  }

  /**
   * 直播接管 - 文字/mp3接管
   *
   * @param {Integer} command - 接管类型（1 - 文字， 2 - mp3）
   */
  handleInterrupt(command = 1) {
    let param;
    // 数据封装
    if (command === 1) {
      param = {
        tts_query: {
          content: this.text,
          ssml: false,
        },
      };
    } else {
      param = {
        audio: audioUrl, // audioUrl - mp3上传后的文件链接
      };
    }

    this.$axios
      .post("/api/2dvh/v2/material/live/channel/command", {
        sessionId: this.sessionId,
        param: JSON.stringify(param),
      })
      .then((res) => {
        const { code, message, data } = res;
        if (code !== 0) {
          this.$message.error(message);
          return;
        }

        // commandTraceId - 可用于队伍插队/取消单次接管
        const { isSuccess, commandTraceId } = data;

        if (isSuccess) {
          this.$message.success("接管成功");
          return;
        }

        this.$message.error("接管失败");
      });
  }

  /**
   * 直播接管 - 麦克风接管socket实例
   *
   */
  initAudioSSE() {
    this.sseAudioClient = new Sse("server");
    this.sseAudioClient.init(
      `/api/live-takeover-cc/v1/material/live/channel/command/microphone?clientType=BROWSER&sessionId=${this.sessionId}&Authorization=${this.sessionToken}`
    );

    this.sseAudioClient.addEventListener("opened", () => {}, false);
    this.sseAudioClient.addEventListener(
      "message",
      (message) => {
        if (message.detail) {
          const code = message.detail.code;
          const msg = message.detail.message;
          if (code === 1) {
            this.$message.error(msg);
            return;
          }

          /***
           * 服务端回传code说明
           * 
           * 1 - 认证相关问题
           * 2 - 【麦克风接管】服务端已就绪
           * 4 - 【麦克风接管】算法侧断连，停止发送二进制pcm
           * 6 - 【麦克风接管】没有对应的算法worker，不发送接管数据
           * 7 - 【麦克风接管】算法端接管冲突被拒绝
           * 8 - 【麦克风接管】服务端正常接收中
           * 18 - 接管结束
           * 19 - 接管开始
           * 20 - 接管入队成功
           * 84115931 - 接管出错，未知原因
           */
          switch (code) {
            case 4:
            case 6:
            case 7:
              // 异常，停止发送语音
              this.closeAudioInterrupt();
              break;
            default:
              console.log(`[${code}]${msg}`);
          }
        }
      },
      false
    );

    this.sseAudioClient.addEventListener(
      "close",
      () => {
        console.log("语音接管已关闭");
      },
      false
    );
  }

  /**
   * 麦克风接管 - 开启麦克风
   */
  openAudioInterrupt() {
    this.audioOnlineInstance = new recorder(this.sseAudioClient);
    this.audioOnlineInstance.start(() => {

      // socket发送开始帧
      this.sseAudioClient.sendMessage(
        JSON.stringify({
          code: 0,
          message: "begin",
        })
      );
    });
  }

  /**
   * 麦克风接管 - 关闭麦克风
   */
  closeAudioInterrupt() {
    this.audioOnlineInstance.stop();

    // socket发送结束帧
    this.sseAudioClient.sendMessage(
      JSON.stringify({
        code: 99,
        message: "eof",
      })
    );
  }

  /**
   * 关闭直播
   *
   */
  handleExit() {
    this.$axios
      .post(`${this.$api.live.close}`, {
        sessionId: this.sessionId,
      })
      .then((res) => {
        const { code, message } = res;
        if (code !== 0) {
          this.$message.error(message);
          return;
        }

        // 关闭声网客户端
        if (this.rtcClient) {
          this.rtcClient.close();
          this.rtcClient = null;
        }

        // 重置状态
        this.sessionId = null;
        this.sessionToken = null;
        this.isConnect = false;
        this.channelId = undefined;
      });
  }
}
